// Copyright (C) Mikko Apo (apo@iki.fi)
// The following code may be used to write free software
// if credit is given to the original author.
// Using it for anything else is not allowed without permission
// from the author.

/*
  1.26 Fixed MuteInertia so that mute actually works with inertia len.
  1.25 Thanks to Discharge for spotting that
		"mute on was not on when loading a song" bug. Fixed now.
  1.23 New attribute: InertiaLength:tick/ms
  1.22 Fixed the problem of champ not obeying the settings when
	   a song was loaded. Thanks to DjLaser for noticing that one.
  1.21 Midi-note-mute accuracy increased.
	   Selectable between tick/realtime(=rt) now.
	   Reduced cpu usage to the level of a basic gain.
  1.2 Midi-note-mute added (thanks to Hymax for asking for it =).
      Three new attributes added for it,
	  menu option and some info in about nox.
  1.1 Added Exp.Inertia and a related attributes. Fixed stereomode.
	  (thanks for the bug report Davide).
  1.0 Initial release
  */


#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <string.h>
#include "../mdk.h"

#define miCOMMAND_STRING "About...\nMidi Mute Note to last Note"
#define miMACHINE_NAME "cheapo amp"
#define miSHORT_NAME "ch.amp"
#define miMACHINE_AUTHOR "Mikko Apo (apo@iki.fi)"
#define miMAX_TRACKS		0
#define miMIN_TRACKS		0
#define miNUMGLOBALPARAMETERS	4
#define miNUMTRACKPARAMETERS	0
#define miNUMATTRIBUTES		15
#define miVERSION "1.26"

//	Parameters

CMachineParameter const paraMaxGain = 
{ pt_word, "Max Gain","Max Gain in dB [0=-96.0dB, 960=0.0dB, 1280=+32.0dB]",0,1280,0xffff,MPF_STATE,960};

CMachineParameter const paraGain = 
{ pt_word, "Gain","Gain in %",0,1000,0xffff,MPF_STATE,1000};

CMachineParameter const paraMute = 
{ pt_switch, "Mute","",0,1,SWITCH_NO,MPF_STATE,0};

CMachineParameter const paraInertia = 
{ pt_byte, "Inertia","Inertia",0,0xfe,0xff,MPF_STATE,10 };

// List of all parameters, track parameters last

CMachineParameter const *pParameters[] = 
{ &paraMaxGain,&paraGain,&paraMute,&paraInertia};

// Attributes

CMachineAttribute const attrMinDB = { "M.Gain neg. dB [min]" ,0,128,96 };
CMachineAttribute const attrMaxDB = { "M.Gain pos. dB [max]" ,0,128,32 };
CMachineAttribute const attrMinPercent = { "Min Gain %" ,0,1000,0 };
CMachineAttribute const attrMaxPercent = { "Max Gain %" ,0,1000,100 };
CMachineAttribute const attrMuteInertia = { "Inertia controlled Mute [0=no]" ,0,1,0 };
CMachineAttribute const attrInertiatype = { "Inertia type [0=lin,1=exp]" ,0,1,0 };
CMachineAttribute const attrMuteOut = { "Mute start fade out [ms]" ,0,10000,50 };
CMachineAttribute const attrMuteIn = { "Mute end fade in [ms]" ,0,10000,50 };
CMachineAttribute const attrExpZeroOut = { "0 level E.Inertia Out [-dB]" ,0,500,220 };
CMachineAttribute const attrExpZeroIn = { "0 level E.Inertia In [-dB]" ,0,500,220 };
CMachineAttribute const attrMMChan = { "MidiMute channel [0=off,17=all]" ,0,17,0 };
CMachineAttribute const attrMMNote= { "MidiMute note" ,0,500,48 };
CMachineAttribute const attrMMMode = { "MidiMuteMode [1=+/-,2=-/+,3=-,+]" ,1,3,3 };
CMachineAttribute const attrMMAccu = { "MidiMute accurary [0=tick,1=rt]" ,0,1,1 };
CMachineAttribute const attrInertiaLen = { "Inertia Len [0 is tick-scale,0< is in ms]" ,0,10000,0 };


// List of all attributes

CMachineAttribute const *pAttributes[] =
{ &attrMinDB,&attrMaxDB,&attrMinPercent,&attrMaxPercent,&attrMuteInertia, \
  &attrInertiatype,&attrMuteOut,&attrMuteIn,&attrExpZeroOut,&attrExpZeroIn, \
  &attrMMMode,&attrMMChan,&attrMMNote,&attrMMAccu,&attrInertiaLen
};

#pragma pack(1)

class gvals
{
public:
	word maxgain;
	word gain;
	byte mute;
	byte inertia;
};

class avals
{
public:
	int dbmin;
	int dbmax;
	int percentmin;
	int percentmax;
	int muteinertia;
	int inertiatype;
	int muteout;
	int mutein;
	int expzeroout;
	int expzeroin;
	int mmmode;
	int mmchan;
	int mmnote;
	int mmaccu;
	int inertialen;
};

#pragma pack()

// Machine's info

CMachineInfo const MacInfo = 
{
	MT_EFFECT,MI_VERSION,MIF_DOES_INPUT_MIXING,miMIN_TRACKS,miMAX_TRACKS,
	miNUMGLOBALPARAMETERS,miNUMTRACKPARAMETERS,pParameters,miNUMATTRIBUTES,pAttributes,
#ifdef _DEBUG
	miMACHINE_NAME" [DEBUG]"
#else
	miMACHINE_NAME
#endif
	,miSHORT_NAME,miMACHINE_AUTHOR,miCOMMAND_STRING
};


class miex : public CMDKMachineInterfaceEx
{

};

class mi : public CMDKMachineInterface
{
public:
	mi();

	public:
	virtual void Command(int const i);
	virtual void Tick();
	virtual char const *DescribeValue(int const param, int const value);

	virtual void MDKInit(CMachineDataInput * const pi);
	virtual bool MDKWork(float *psamples, int numsamples, int const mode);
	virtual bool MDKWorkStereo(float *psamples, int numsamples, int const mode);
	virtual void MDKSave(CMachineDataOutput * const po) { }

	virtual CMDKMachineInterfaceEx *GetEx() { return &ex; }
	virtual void OutputModeChanged(bool stereo) {}

	virtual void AttributesChanged();
	virtual void MidiNote(int const channel, int const value, int const velocity);


	public:
	miex ex;
	gvals gval;
	avals aval;

private:

	void paramchanged(bool gainchanged,int mutevalue);
	int inertiatime(int value);
	double inertiaunits(int value);

	double valMaxGain;
	float valGain;
	bool valMute;
	long valInertia,valInertiaSamples;
	long valMuteInertia,valMuteIn,valMuteOut,valDBMin,valDBMax,valPercentMin,valPercentMax;
	double amp_current;
	double amp_dest,amp_inc;	
	long inc_counter;
	bool inertia;
	bool firsttime;
	int valInertiaType,valInertiaLen;
	double valExpZeroIn,valExpZeroOut;
	int lastnotein,lastchanin;
	CMachine *thismachine;
	int mmvalue;
};


DLL_EXPORTS

mi::mi()
{
	GlobalVals = &gval;
	AttrVals = (int *)&aval;
}

char *notestring(int value)
{
	static char buf[5]={0,0,0,0,0};
	char notes[][3]={"C","C#","D","D#","E","F","F#","G","G#","A","A#","B"};
    sprintf(buf,"%s%d",notes[value%12],value/12-1);
	return buf;
}

void mi::Command(int const i)
{
	char txt[1000];
	switch(i)
	{
	case 0:
		sprintf(txt,miMACHINE_NAME"\n\nBuild date: "__DATE__"\nVersion: "miVERSION"\nCode by: "miMACHINE_AUTHOR"\n\nCheck out http://www.iki.fi/apo/buzz/\nfor more buzz stuff.\n\nExcellent skin made by Hymax.\n");
		sprintf(txt,"%s\nMidi mute channel: ",txt);
		switch(aval.mmchan)
		{
		case 0:
			sprintf(txt,"%sdisabled",txt);
			break;
		case 17:
			sprintf(txt,"%sall",txt);
			break;
		default:
			sprintf(txt,"%s%d",txt,aval.mmchan);
			break;
		}
		sprintf(txt,"%s\nMidi mute note: %d (%s)\nMidi mute behaviour: %d. %s\n",txt,aval.mmnote,notestring(aval.mmnote),aval.mmmode,(aval.mmmode==1)?"note on->mute on, note off->mute off":(aval.mmmode==2)?"note on->mute off,note off->mute on":"note on->change to mute/unmute");
		if(lastnotein!=-1)
		{
		  sprintf(txt,"%sLast midi note recieved: %d (%s) from channel %d.\n",txt,lastnotein,notestring(lastnotein),lastchanin);
		}
		pCB->MessageBox(txt);
		break;
	case 1:
		if(lastnotein!=-1)
		{
		  aval.mmnote=lastnotein;
		  aval.mmchan=lastchanin;
		}
		break;
	}
}

void mi::AttributesChanged()
{
	long tmp;
	valMuteInertia=aval.muteinertia;
	valMuteIn=(int)((aval.mutein*pMasterInfo->SamplesPerSec)/1000);
	valMuteOut=(int)((aval.muteout*pMasterInfo->SamplesPerSec)/1000);
	valDBMin=aval.dbmin;
	valDBMax=aval.dbmax;
	valPercentMin=aval.percentmin;
	valPercentMax=aval.percentmax;
	if(valPercentMin>valPercentMax)
	{
		tmp=valPercentMin;
		valPercentMin=valPercentMax;
		valPercentMax=tmp;
	}
	if(valPercentMin==valPercentMax)
	{
		valPercentMin=attrMinPercent.DefValue;
		valPercentMax=attrMaxPercent.DefValue;
	}
	if(!valDBMin&&!valDBMax)
	{
		valDBMin=attrMinDB.DefValue;
		valDBMax=attrMaxDB.DefValue;
	}
	valInertiaType=aval.inertiatype;
	if(valInertiaLen!=aval.inertialen)
	{
		valInertiaLen=aval.inertialen;
		valInertiaSamples=inertiatime(valInertia);
	}
	valExpZeroOut=pow(10,-aval.expzeroout*0.05);
	valExpZeroIn=pow(10,-aval.expzeroin*0.05);
}

double mi::inertiaunits(int value)
{
	double tmp;
	if(valInertiaLen==0)
	{
		tmp =value/10.0;
#define vr(x,y,z) if(value>(x)) tmp=(y)+(z)*(value-(x))
		vr(80,8,1);
		vr(136,64,2);
		vr(232,256,64);
		vr(236,512,128);
		vr(240,1024,256);
		vr(244,2048,512);
		vr(248,4096,1024);
#undef vr
		return tmp;
	}
	return ((double)value*valInertiaLen)/paraInertia.MaxValue;
}

int mi::inertiatime(int value)
{
	if(valInertiaLen==0)
	{
		return (int)(inertiaunits(value)*pMasterInfo->SamplesPerTick);
	}
	return (int)(inertiaunits(value)*pMasterInfo->SamplesPerSec/1000.0);
}

char const *mi::DescribeValue(int const param, int const value)
{
	static char txt[100];

	if(!param) sprintf(txt,"%-2.1f dB",(float) (value*(valDBMin+valDBMax)/1280.0-valDBMin) );
	if(param==1) sprintf(txt,"%.1f %%",value*(aval.percentmax-aval.percentmin)/1000.0+aval.percentmin);
	if(param==2) sprintf(txt,"%s",value?"on":"off");
	if(param==3)
	{
		sprintf(txt,"%.1f %s",inertiaunits(value),(valInertiaLen)?"ms":"ticks");
	}

	return txt;
}

#define calc_maxgain(value) pow(10, ( (value*(valDBMin+valDBMax)/(float)paraMaxGain.MaxValue-valDBMin) * 0.05))
#define calc_gain(value) (float)(((value)*(aval.percentmax-aval.percentmin)/1000.0+aval.percentmin)/100.0)

void mi::MDKInit(CMachineDataInput * const pi)
{
	thismachine=pCB->GetThisMachine();
	AttributesChanged();
	valMaxGain=calc_maxgain(paraMaxGain.DefValue);
	valGain=calc_gain(paraGain.DefValue);
	valMute=(paraMute.DefValue)?true:false;
	valInertia=paraInertia.DefValue;
	valInertiaSamples=inertiatime(valInertia);
	amp_current=1.0;
	inertia=false;
	firsttime=true;
	lastnotein=-1;
	mmvalue=-1;
}

void mi::MidiNote(int const channel, int const value, int const velocity)
{
	int action=-1;

	lastnotein=value;
	lastchanin=channel+1;
	// we need to return if
	// - mmchan = 0 (disabled)
	// - mmchan != 17 and channel != mmchan-1
	// - note is not the wanted one
    if(aval.mmchan == 0 || value != aval.mmnote || channel != aval.mmchan-1 && aval.mmchan != 17 )
		return;

	if(velocity>0) // note played
	{
		switch(aval.mmmode)
		{
		case 1:
			action=1;	// on
			break;
		case 2:
			action=0;	// off
			break;
		case 3:
			if(aval.mmaccu==0&&mmvalue!=-1)
			{
			  action=(mmvalue==1)?0:1;	// there was a previous change during this same tick
			} else
			{
			  action=(valMute)?0:1;	// toggle
			}
			break;
		}
	} else // note released
	{
		switch(aval.mmmode)
		{
		case 1:
			action=0;	// off
			break;
		case 2:
			action=1;	// on
			break;
		case 3:
			break;
		}
	}
	if(action!=-1)
	{
		pCB->ControlChange(thismachine,1,0,2,action);
		if(aval.mmaccu==1)
		{
			paramchanged(false,action);
		}
	}
 // if we did something, store it, but don't overwrite previous action if there was no action
	if(action!=-1)
	  mmvalue=action;
}

void mi::paramchanged(bool gainchanged,int mutevalue)
{
	bool mutechanged=false;

	if(mutevalue==1&&valMute==false||mutevalue==0&&valMute==true)
	{
		 // check if the wanted mute action is the same as the previous midimute action
		if(mmvalue=-1||valMute==true&&mmvalue==0||valMute==false&&mmvalue==1)
		{
			// it wasn't the same, we can change

			if(mutevalue==1)
			{
				valMute=true;
			}
			if(mutevalue==0)
			{
				valMute=false;
			}
			mutechanged=true;
		}
	}

	// mute overrides gain changes
  if(mutechanged)
  {
	  if(valMute) // the channel is going to be muted
	  {
		  amp_dest=0.0;
		  inc_counter=valMuteInertia?valInertiaSamples:valMuteIn;
	  } else  // the channel is going to be unmuted
	  {
		  amp_dest=valGain*valMaxGain;
		  inc_counter=valMuteInertia?valInertiaSamples:valMuteOut;
	  }
  } else if(gainchanged)
  {
  // mute over rides gain changes
	  if(valMute)
	  {
		  gainchanged=false;
	  } else
	  {
		if(valGain*valMaxGain!=amp_dest) // if the destination is the same, don't trigger new inertia
		{
	      amp_dest=valGain*valMaxGain;	// trigger new inertia
	      inc_counter=valInertiaSamples;
		} else
		{
			gainchanged=false;		// nope, it was a false alarm (repeated value)
		}
	  }
  }
  if(firsttime)
  {
	  if(valMute)
	  {
		  amp_current=0;
	  } else
	  {
		amp_current=valGain*valMaxGain; // use these values after loading
	  }
	  inertia=false;
	  firsttime=false;
  } else
  {
	if(mutechanged||gainchanged)
	{
		switch(valInertiaType)
		{
		case 0:
				amp_inc=(amp_dest-amp_current)/inc_counter;
			break;
		case 1:
			bool dest=false,curr=false;
// amp_current*amp_inc^inc_counter=amp_dest
// amp_inc^inc_counter=amp_dest/amp_current || log10()
// log10(amp_inc)=1/inc_counter*log10(amp_dest/amp_current) || 10^
// amp_inc=10^(log10(amp_dest/amp_current)/inc_counter)
			if(amp_dest==0.0)
			{
				dest=true;
				amp_dest=valExpZeroOut;
			}
			if(amp_current==0.0) 
			{
				curr=true;
				amp_current=valExpZeroIn;
			}
				amp_inc=pow(10, (log10(amp_dest/amp_current)/inc_counter) );
			if(dest) amp_dest=0.0;
			break;
		}
		  inertia=true;
	  }
  }
}

void mi::Tick()
{
  bool gainchanged=false;
  int mutevalue=-1;

  if (gval.maxgain != paraMaxGain.NoValue)
  {
	valMaxGain=calc_maxgain(gval.maxgain);
	gainchanged=true;
  }
  if (gval.gain != paraGain.NoValue)
  {
	valGain=calc_gain(gval.gain);
	gainchanged=true;
  }
  if (gval.mute != paraMute.NoValue)
  {
	mutevalue=gval.mute;
  }
  if(gval.inertia != paraInertia.NoValue)
  {
	  valInertia=gval.inertia;
	  valInertiaSamples=inertiatime(valInertia);
  }

  paramchanged(gainchanged,mutevalue);
  mmvalue=-1;

} 

// update inertia counters
#define upd_inertia	if(inertia) { \
			inc_counter-=numsamples; \
			if(inc_counter>0) \
			amp_current=(valInertiaType)?(amp_current*pow(amp_inc,numsamples)):(amp_current+amp_inc*numsamples); \
			else {inertia=false;amp_current=amp_dest;} \
		}

#define lin_inertia if(inertia) { \
			inc_counter--; \
			if(inc_counter>0) amp_current+=amp_inc; \
			else {inertia=false;amp_current=amp_dest;} \
		}

#define exp_inertia	if(inertia) { \
			inc_counter--; \
			if(inc_counter>0) amp_current*=amp_inc; \
			else {inertia=false;amp_current=amp_dest;} \
		}

bool mi::MDKWorkStereo(float *psamples, int numsamples, int const mode)
{
	if (mode==WM_NOIO)
	{
		return false;
	}

	if (mode==WM_WRITE) // no sound coming in
	{
		// if there's inertia, update counters
		upd_inertia;
		return false;
	}

	if (mode == WM_READ)		// <thru>
	{
		// if there's inertia, update counters
		upd_inertia;
		return true;
	}

	// no sound coming out if amp_current = 0 and there's no inertia
	if(amp_current==0.0&&inertia==false) return false;

	if(inertia==false)
	{
		do 
		{
			psamples[0]=(float)(amp_current*psamples[0]);
			psamples[1]=(float)(amp_current*psamples[1]);
			psamples+=2;
		} while(--numsamples);
	} else
	{
		// here we have the special inertia handling code
	switch(valInertiaType)
	{
	case 0:
	do 
	{
		lin_inertia;
		psamples[0]=(float)(amp_current*psamples[0]);
		psamples[1]=(float)(amp_current*psamples[1]);
		psamples+=2;
	} while(--numsamples);
	break;
	case 1:
	do 
	{
		exp_inertia;
		psamples[0]=(float)(amp_current*psamples[0]);
		psamples[1]=(float)(amp_current*psamples[1]);
		psamples+=2;
	} while(--numsamples);
	break;
	}
	}
	return true;
}
bool mi::MDKWork(float *psamples, int numsamples, int const mode)
{
	if (mode==WM_NOIO)
	{
		return false;
	}

	if (mode==WM_WRITE) // no sound coming in
	{
		// if there's inertia, update counters
		upd_inertia;
		return false;
	}

	if (mode == WM_READ)		// <thru>
	{
		// if there's inertia, update counters
		upd_inertia;
		return true;
	}

	// no sound coming out if amp_current = 0 and there's no inertia
	if(amp_current==0.0&&!inertia) return false;

	if(inertia==false)
	{
		do 
		{
			psamples[0]=(float)(amp_current*psamples[0]);
			psamples++;
		} while(--numsamples);
	} else
	{
		// here we have the special inertia handling code
	switch(valInertiaType)
	{
	case 0:
	do 
	{
		lin_inertia;
		psamples[0]=(float)(amp_current*psamples[0]);
		psamples++;
	} while(--numsamples);
	break;
	case 1:
	do 
	{
		exp_inertia;
		psamples[0]=(float)(amp_current*psamples[0]);
		psamples++;
	} while(--numsamples);
	break;
	}
	}
	return true;
}

#undef upd_inertia